<?php

namespace App\Http\Controllers;

use App\Models\Pelanggan;
use Illuminate\Http\Request;

class PelangganController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Pelanggan::query();

        // Filter by name
        if ($request->filled('search')) {
            $query->where('nama_pelanggan', 'like', '%' . $request->search . '%');
        }

        // Filter by date range
        if ($request->filled('start_date')) {
            $query->whereDate('tanggal', '>=', $request->start_date);
        }

        if ($request->filled('end_date')) {
            $query->whereDate('tanggal', '<=', $request->end_date);
        }

        // Calculate total income from the filtered query
        $totalPendapatan = $query->sum('total_bayar');

        // Get the results, ordered by the latest, and paginate
        $pelanggans = $query->latest()->paginate(10)->appends($request->query());

        return view('pelanggan.index', compact('pelanggans', 'totalPendapatan'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('pelanggan.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'nama_pelanggan' => 'required|string|max:255',
            'tanggal' => 'required|date',
            'alamat' => 'required|string',
            'no_telepon' => 'required|string|max:20',
            'merk_frame' => 'required|string|max:255',
            'total_bayar' => 'required|numeric|min:0',
            'dp' => 'required|numeric|min:0',
            'tanggal_kirim' => 'nullable|date',
        ]);

        $data = $request->all();
        $data['sisa_pembayaran'] = $data['total_bayar'] - $data['dp'];

        Pelanggan::create($data);

        return redirect()->route('pelanggan.index')
                         ->with('success', 'Data pelanggan berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Pelanggan $pelanggan)
    {
        return view('pelanggan.show', compact('pelanggan'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Pelanggan $pelanggan)
    {
        return view('pelanggan.edit', compact('pelanggan'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Pelanggan $pelanggan)
    {
        $request->validate([
            'nama_pelanggan' => 'required|string|max:255',
            'tanggal' => 'required|date',
            'alamat' => 'required|string',
            'no_telepon' => 'required|string|max:20',
            'merk_frame' => 'required|string|max:255',
            'total_bayar' => 'required|numeric|min:0',
            'dp' => 'required|numeric|min:0',
            'tanggal_kirim' => 'nullable|date',
        ]);

        $data = $request->all();
        $data['sisa_pembayaran'] = $data['total_bayar'] - $data['dp'];

        $pelanggan->update($data);

        return redirect()->route('pelanggan.index')
                         ->with('success', 'Data pelanggan berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Pelanggan $pelanggan)
    {
        $pelanggan->delete();

        return redirect()->route('pelanggan.index')
                         ->with('success', 'Data pelanggan berhasil dihapus.');
    }
}
